//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.0

import wesual.Controls 1.0


PopupAnchorItem {
    id : popup

    property Item dropdownButton
    property list<Action> actions
    property int activeIndex : -1

    signal accepted(int result)

    popup : Item {
        id : dropDown

        // animation init
        width : 0
        height : 0
        opacity : 0
        clip : true

        Rectangle {
            id : bg

            anchors.fill : parent
            border {
                color : UI.color(UI.PrimaryControlBorder)
                width : 1
            }
        }

        // Key handling
        Keys.onUpPressed : {
            if (popup.actions.length < 1)
                return;

            var index = popup.activeIndex - 1;
            if (index < 0)
                index = popup.actions.length - 1;
            popup.activeIndex = index;
        }
        Keys.onDownPressed : {
            if (popup.actions.length < 1)
                return;

            popup.activeIndex = (popup.activeIndex + 1) % popup.actions.length;
        }
        Keys.onEscapePressed : popup.reject()
        Keys.onReturnPressed :
            popup.accepted(popup.activeIndex >= 0 ?
                           popup.activeIndex : undefined)

        Column {
            id : itemColumn

            readonly property int margin : 8

            readonly property int idealWidth : {
                var max = popup.dropdownButton ?
                          (popup.dropdownButton.width - bg.border.width * 2) : 0;
                for (var i = 0; i < children.length; ++i) {
                    max = Math.max(max, children[i].implicitWidth);
                }

                return max;
            }

            x : bg.border.width
            y : bg.border.width
            width : idealWidth
            height : childrenRect.height

            // 3px space before first element
            Item {
                width   : itemColumn.idealWidth
                height  : 3
            }

            Repeater {
                model : popup.actions
                delegate : Item {

                    readonly property int itemIndex : index
                    readonly property bool active : popup.activeIndex === index

                    implicitWidth : label.implicitWidth + 2 * itemColumn.margin
                    width : itemColumn.idealWidth
                    height : 26

                    Rectangle {
                        id : itemBackground

                        height : parent.height
                        width : itemColumn.idealWidth
                        color : {
                            if (active && mouseArea.pressed)
                                return UI.color(UI.PrimaryPress);
                            else if (active)
                                return UI.color(UI.PrimaryHover);
                            else
                                return UI.color(UI.PrimaryBase);
                        }
                        Behavior on color {
                            ColorAnimation {
                                duration : 0
                            }
                        }
                    }

                    Text {
                        id : label
                        text : modelData.text
                        x : itemColumn.margin
                        y : 19 - font.pixelSize
                        anchors.verticalCenter : parent.verticalCenter
                        font {
                            family : UI.fontFamily(UI.PrimaryFont)
                            weight : UI.fontWeight(UI.PrimaryFont)
                            pixelSize : 14
                        }
                    }
                }
            }

            // 3px space after last element
            Item {
                width   : itemColumn.idealWidth
                height  : 3
            }
        }

        MouseArea {
            id : mouseArea
            anchors.fill : parent
            acceptedButtons : Qt.LeftButton
            hoverEnabled : true
            onPositionChanged : {
                var child = itemColumn.childAt(mouse.x, mouse.y);
                if (child && child.itemIndex !== undefined) {
                    popup.activeIndex = child.itemIndex;
                }
            }
            onReleased : popup.accepted(popup.activeIndex)
        }

        states : [
            State {
                name : "visible"
                when : popup.popupVisible

                PropertyChanges {
                    target : dropDown
                    opacity : 1
                    height : itemColumn.height + 2 * bg.border.width
                    width  : itemColumn.width  + 2 * bg.border.width
                }
            }
        ]

        transitions : Transition {
            PropertyAnimation {
                properties : "width, height, opacity"
                duration : Math.min(itemColumn.height, 200)
                easing.type : Easing.OutElastic
            }
        }
    }
}
